package layaair.game.browser;

import android.content.Context;
import android.graphics.Color;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.Surface;
import android.view.TextureView;
import android.widget.AbsoluteLayout;
import android.widget.FrameLayout;

import layaair.game.conch.LayaConch5;
import java.io.File;
import java.io.FileOutputStream;

public class LayaVideoPlayer extends FrameLayout {


    private long mNativePtr;
    private MediaPlayer mPlayer;
    private TextureView mTextureView;
    private Surface mSurface;
    private boolean mPrepared = false;
    private boolean mAutoplay = false;
    private boolean mLoop = false;
    private boolean mMuted = false;
    private float mVolume = 1.0f;
    private float mRate = 1.0f;
    private String mUrl;
    private String mTempFilePath;
    // ObjectFit: 0=fill, 1=contain, 2=cover
    private int mObjectFit = 1;
    private int mVideoW = 0;
    private int mVideoH = 0;
    private boolean mControls = true;
    private boolean mAwaitFirstFramePause = false; // 当不自动播放时，拉起首帧后立刻暂停

    private int mX, mY, mW, mH;
    private final Handler mMainHandler = new Handler(Looper.getMainLooper());

    public LayaVideoPlayer(long nativePtr) {
        super(getAppContext());
        mNativePtr = nativePtr;
        setupView();
    }

    private static Context getAppContext() {
        return LayaConch5.GetInstance().getGameContext();
    }

    private void runOnUiThread(Runnable r) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            r.run();
        } else {
            mMainHandler.post(r);
        }
    }

    private void setupView() {
        runOnUiThread(() -> {
            // 确保子内容不会溢出父视图边界
            try { setBackgroundColor(Color.BLACK); } catch (Throwable ignore) {}
            mTextureView = new TextureView(getContext());
            mTextureView.setOpaque(true);
            try { mTextureView.setBackgroundColor(Color.BLACK); } catch (Throwable ignore) {}
            // 移除 SDK21+ 的剪裁与轮廓设置，保持默认矩形裁剪
            mTextureView.setSurfaceTextureListener(new TextureView.SurfaceTextureListener() {
                @Override public void onSurfaceTextureAvailable(android.graphics.SurfaceTexture surface, int width, int height) {
                    try {
                        if (mSurface != null) try { mSurface.release(); } catch (Throwable ignore) {}
                        mSurface = new Surface(surface);
                        if (mPlayer != null) mPlayer.setSurface(mSurface);
                    } catch (Throwable ignore) {}
                }
                @Override public void onSurfaceTextureSizeChanged(android.graphics.SurfaceTexture surface, int width, int height) {}
                @Override public boolean onSurfaceTextureDestroyed(android.graphics.SurfaceTexture surface) { return true; }
                @Override public void onSurfaceTextureUpdated(android.graphics.SurfaceTexture surface) {
                    // 首帧到达后立即暂停，实现 autoplay=false 时显示第一帧
                    if (mAwaitFirstFramePause) {
                        mAwaitFirstFramePause = false;
                        try { if (mPlayer != null && mPrepared) mPlayer.pause(); } catch (Throwable ignore) {}
                    }
                }
            });
            addView(mTextureView, new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

            AbsoluteLayout parent = LayaConch5.GetInstance().getAbsLayout();
            if (parent != null) {
                AbsoluteLayout.LayoutParams lp = new AbsoluteLayout.LayoutParams(1, 1, 0, 0);
                parent.addView(this, lp);
                setVisibility(VISIBLE);
            }
        });
    }

    // JNI API
    public void load(String url) {
        mUrl = url;
        cleanupTempFile();
        releasePlayer();
        try {
            mPlayer = new MediaPlayer();
            mPlayer.setLooping(mLoop);
            if (mSurface != null) mPlayer.setSurface(mSurface);
            mPlayer.setOnPreparedListener(mp -> {
                mPrepared = true;
                applyVolumeAndRate();
                try {
                    mVideoW = mp.getVideoWidth();
                    mVideoH = mp.getVideoHeight();
                    // 尺寸就绪后重新应用布局与变换
                    applyLayout();
                } catch (Throwable ignore) {}
                emit(mNativePtr, "loadedmetadata");
                if (mAutoplay) {
                    play();
                } else {
                    // 拉起首帧后立刻暂停
                    try { setVisibility(VISIBLE); } catch (Throwable ignore) {}
                    mAwaitFirstFramePause = true;
                    try { mPlayer.seekTo(0, MediaPlayer.SEEK_CLOSEST); } catch (Throwable ignore) {}
                    try { mPlayer.start(); } catch (Throwable ignore) {}
                }
            });
            mPlayer.setOnCompletionListener(mp -> emit(mNativePtr, "ended"));
            mPlayer.setOnErrorListener((mp, what, extra) -> {
                emit(mNativePtr, "error:" + what + "/" + extra);
                return true;
            });
            mPlayer.setDataSource(mUrl);
            mPlayer.prepareAsync();
        } catch (Throwable t) {
            emit(mNativePtr, "error:" + t.getMessage());
        }
    }

    public void loadBuffer(String url, byte[] data) {
        cleanupTempFile();
        releasePlayer();
        try {
            // create temp file with extension guessed from url
            String ext = guessExtFromUrl(url);
            File temp = File.createTempFile("video_", ext, getContext().getCacheDir());
            mTempFilePath = temp.getAbsolutePath();
            if (data != null) {
                FileOutputStream fos = new FileOutputStream(temp);
                fos.write(data);
                fos.close();
            }

            mPlayer = new MediaPlayer();
            mPlayer.setLooping(mLoop);
            if (mSurface != null) mPlayer.setSurface(mSurface);
            mPlayer.setOnPreparedListener(mp -> {
                mPrepared = true;
                applyVolumeAndRate();
                try {
                    mVideoW = mp.getVideoWidth();
                    mVideoH = mp.getVideoHeight();
                    // 尺寸就绪后重新应用布局与变换
                    applyLayout();
                } catch (Throwable ignore) {}
                emit(mNativePtr, "loadedmetadata");
                if (mAutoplay) {
                    play();
                } else {
                    try { setVisibility(VISIBLE); } catch (Throwable ignore) {}
                    mAwaitFirstFramePause = true;
                    try { mPlayer.seekTo(0, MediaPlayer.SEEK_CLOSEST); } catch (Throwable ignore) {}
                    try { mPlayer.start(); } catch (Throwable ignore) {}
                }
            });
            mPlayer.setOnCompletionListener(mp -> emit(mNativePtr, "ended"));
            mPlayer.setOnErrorListener((mp, what, extra) -> { emit(mNativePtr, "error:" + what + "/" + extra); return true; });
            mPlayer.setDataSource(getContext(), Uri.fromFile(temp));
            mPlayer.prepareAsync();
        } catch (Throwable t) {
            emit(mNativePtr, "error:" + t.getMessage());
        }
    }

    public void play() {
        try {
            runOnUiThread(() -> {
                setVisibility(VISIBLE);
                try { if (mPlayer != null && mPrepared) mPlayer.start(); } catch (Throwable ignore2) {}
            });
        } catch (Throwable ignore) {}
    }

    public void pause() {
        try { if (mPlayer != null && mPrepared) mPlayer.pause(); } catch (Throwable ignore) {}
    }

    public void Stop() {
        runOnUiThread(() -> {
            releasePlayer();
            try { setVisibility(GONE); } catch (Throwable ignore) {}
        });
    }

    public boolean isPaused() {
        try { return mPlayer != null && mPrepared && !mPlayer.isPlaying(); } catch (Throwable t) { return true; }
    }

    public boolean isLoop() { return mLoop; }
    public void setLoop(boolean loop) {
        mLoop = loop;
        try { if (mPlayer != null) mPlayer.setLooping(loop); } catch (Throwable ignore) {}
    }

    public boolean isAutoplay() { return mAutoplay; }
    public void setAutoplay(boolean ap) { mAutoplay = ap; }

    // 一次性设置矩形，避免多次 applyLayout 造成中间态
    public void setRect(double x, double y, double w, double h) {
        mX = (int)Math.round(x);
        mY = (int)Math.round(y);
        mW = (int)Math.round(w);
        mH = (int)Math.round(h);
        //Log.d(TAG, "SetRect x=" + mX + ", y=" + mY + ", w=" + mW + ", h=" + mH);
        applyLayout();
    }

    public int getVideoWidth() { try { return (mPlayer!=null)? mPlayer.getVideoWidth():0; } catch (Throwable t){ return 0; } }
    public int getVideoHeight() { try { return (mPlayer!=null)? mPlayer.getVideoHeight():0; } catch (Throwable t){ return 0; } }
    public int tell() { try { return (mPlayer!=null && mPrepared)? mPlayer.getCurrentPosition():0; } catch (Throwable t){ return 0; } }
    public void seek(int ms) { try { if (mPlayer!=null && mPrepared) mPlayer.seekTo(Math.max(0, ms), MediaPlayer.SEEK_CLOSEST); } catch (Throwable ignore) {} }
    public int getReadyState() { return mPrepared ? 4 : 0; }

    // Rect controls
    public void setX(double x) { mX = (int)Math.round(x); applyLayout(); }
    public void setY(double y) { mY = (int)Math.round(y); applyLayout(); }
    public void setWidth(double w) { mW = (int)Math.round(w); applyLayout(); }
    public void setHeight(double h) { mH = (int)Math.round(h); applyLayout(); }

    // audio params
    public void setMuted(boolean m) { mMuted = m; applyVolumeAndRate(); }
    public void setVolume(float v) { mVolume = v; applyVolumeAndRate(); }
    public void setRate(float r) { mRate = (r<=0f?1f:r); applyVolumeAndRate(); }
    public void setObjectFit(int fit) {
        mObjectFit = (fit < 0 || fit > 2) ? 1 : fit;
        applyLayout();
    }

    // 控件显示开关（占位实现，后续可根据需要添加原生控制栏）
    public void setControls(boolean controls) { mControls = controls; }
    public boolean getControls() { return mControls; }

    private void applyVolumeAndRate() {
        try {
            if (mPlayer != null) {
                float vol = mMuted ? 0f : Math.max(0f, Math.min(1f, mVolume));
                mPlayer.setVolume(vol, vol);
                if (Build.VERSION.SDK_INT >= 23) {
                    try {
                        mPlayer.setPlaybackParams(mPlayer.getPlaybackParams().setSpeed(mRate));
                    } catch (Throwable ignore) {}
                }
            }
        } catch (Throwable ignore) {}
    }

    private void applyLayout() {
        try {
            runOnUiThread(() -> {
                AbsoluteLayout parent = LayaConch5.GetInstance().getAbsLayout();
                if (parent == null) return;
                int width = (mW > 0) ? mW : 1;
                int height = (mH > 0) ? mH : 1;
                // contain/cover 的等比缩放由 MediaPlayer/TextureView 处理，容器保持传入的 width/height
                AbsoluteLayout.LayoutParams lp = new AbsoluteLayout.LayoutParams(width, height, mX, mY);
                try { setLayoutParams(lp); } catch (Throwable ignore) {}
                try { bringToFront(); } catch (Throwable ignore) {}
                try { requestLayout(); invalidate(); } catch (Throwable ignore) {}
                // 移除显式 setClipBounds，常规矩形场景不需要
                // 应用内容缩放到 TextureView：使用其变换矩阵实现 contain/cover/fill
                applyTextureViewTransform(width, height);
            });
        } catch (Throwable ignore) {}
    }

    private void applyTextureViewTransform(int viewW, int viewH) {
        try {
            if (mTextureView == null || mVideoW <= 0 || mVideoH <= 0 || viewW <= 0 || viewH <= 0) return;
            android.graphics.Matrix matrix = new android.graphics.Matrix();
            float vw = viewW;
            float vh = viewH;
            float sx = vw / (float)mVideoW;
            float sy = vh / (float)mVideoH;
            float px = vw * 0.5f;
            float py = vh * 0.5f;
            switch (mObjectFit) {
                case 0: // fill: 拉伸
                    // 直接使用单位矩阵，TextureView 将内容拉伸到控件大小
                    matrix.reset();
                    break;
                case 1: // contain: 等比全可见
                {
                    // 目标等比缩放
                    float target = Math.min(sx, sy);
                    // 相对默认 fill 的缩放（抵消 fill 的非等比）
                    float rx = target / sx;
                    float ry = target / sy;
                    matrix.reset();
                    matrix.setScale(rx, ry, px, py); // 以中心为轴，等比缩放
                    break;
                }
                case 2: // cover: 等比铺满（裁剪）
                default:
                {
                    float target = Math.max(sx, sy);
                    float rx = target / sx;
                    float ry = target / sy;
                    matrix.reset();
                    matrix.setScale(rx, ry, px, py); // 以中心为轴，等比铺满
                    break;
                }
            }
            mTextureView.setTransform(matrix);
            try { mTextureView.invalidate(); } catch (Throwable ignore) {}
        } catch (Throwable ignore) {}
    }

    public void destroy() {
        try {
            runOnUiThread(() -> {
                releasePlayer();
                AbsoluteLayout parent = LayaConch5.GetInstance().getAbsLayout();
                if (parent != null) try { parent.removeView(this); } catch (Throwable ignore) {}
            });
        } catch (Throwable ignore) {}
    }

    private void releasePlayer() {
        try { if (mPlayer != null) { mPlayer.stop(); mPlayer.release(); } } catch (Throwable ignore) {}
        try { if (mSurface != null) { mSurface.release(); } } catch (Throwable ignore) {}
        mPlayer = null; mSurface = null; mPrepared = false;
        cleanupTempFile();
    }

    private static native void emit(long nativePtr, String evt);

    private void cleanupTempFile() {
        try {
            if (mTempFilePath != null) {
                File f = new File(mTempFilePath);
                if (f.exists()) {
                    //noinspection ResultOfMethodCallIgnored
                    f.delete();
                }
            }
        } catch (Throwable ignore) {} finally {
            mTempFilePath = null;
        }
    }

    private static String guessExtFromUrl(String url) {
        try {
            if (url == null) return ".tmp";
            String u = url;
            int q = u.indexOf('?'); if (q >= 0) u = u.substring(0, q);
            int h = u.indexOf('#'); if (h >= 0) u = u.substring(0, h);
            int slash = u.lastIndexOf('/');
            int dot = u.lastIndexOf('.');
            if (dot > slash && dot >= 0 && dot < u.length() - 1) {
                String ext = u.substring(dot);
                if (ext.length() <= 8 && ext.matches("\\.[A-Za-z0-9]+")) return ext;
            }
        } catch (Throwable ignore) {}
        return ".tmp";
    }
}


